﻿/*
 * 以Tuner說明interface的繼承
 * 8/30/2010
 */

using System;
using System.Diagnostics;

namespace Tuner
{
    interface SwitchableDevice
    {
        void TurnOn();
        void TurnOff();
    }
    enum DeviceStatus
    {
        OFF = 0,
        ON = 1
    }
    enum ButtonStatus
    {
        RELEASED = 0,
        PRESSED = 1
    }
    enum QuantityName
    {
        FREQUENCY = 0,
        VOLUME = 1
    }
    interface AdjustableDevice : SwitchableDevice
    {
        void SetLevel(double level, QuantityName qtyName);
    }
    class TunerUnit : AdjustableDevice
    {
        private DeviceStatus status = DeviceStatus.OFF;
        private double frequency;
        private double volume;
        public void TurnOn()
        {
            status = DeviceStatus.ON;
        }
        public void TurnOff()
        {
            status = DeviceStatus.OFF;
        }
        public void SetLevel(double level, QuantityName qtyName)
        {
            if (qtyName == QuantityName.FREQUENCY)
            {
                frequency = level;
            }
            if (qtyName == QuantityName.VOLUME)
            {
                volume = level;
            }
        }
        public bool PowerOn()
        {
            return (status == DeviceStatus.ON);
        }
        public double Frequecy
        {
            get{ return frequency; }
        }
        public double Volume
        {
            get{ return volume; }
        }
    }
    class Button
    {
        private ButtonStatus status = ButtonStatus.RELEASED;
        private SwitchableDevice device;
        public Button(SwitchableDevice device)
        {
            this.device = device;
        }
        public ButtonStatus Status
        {
            get { return status; }
        }
        public void Press()
        {
            if (status == ButtonStatus.RELEASED)
            {
                status = ButtonStatus.PRESSED;
                device.TurnOn();
            }
        }
        public void Release()
        {
            if (status == ButtonStatus.PRESSED)
            {
                status = ButtonStatus.RELEASED;
                device.TurnOff();
            }
        }
    }
    class AdjustingWheel
    {
        private const double TWO_PI = 2.0*3.141592653;
        private double anglePosition = 0.0;
        private AdjustableDevice adjustableDevice;
        private QuantityName qtyName;
        private double qtyMin;
        private double qtyMax;
        public AdjustingWheel(AdjustableDevice adjustableDevice,
            QuantityName qtyName, double qtyMin, double qtyMax)
        {
            this.adjustableDevice = adjustableDevice;
            this.qtyName = qtyName;
            this.qtyMin = qtyMin;
            this.qtyMax = qtyMax;
        }
        public void Turn(double angle)
        {
            anglePosition += angle;
            int n = (int) ( anglePosition / TWO_PI );
            anglePosition -= n * TWO_PI;
            double level = qtyMin + 
                (qtyMax - qtyMin) * anglePosition / TWO_PI;
            adjustableDevice.SetLevel(level, qtyName);
        }
    }
    class AdjustingButton
    {
        private double currentLevel = 0.0;
        private AdjustableDevice adjustableDevice;
        private QuantityName qtyName;
        private double qtyMin;
        private double qtyMax;
        private double deltaQty;
        public AdjustingButton(AdjustableDevice adjustableDevice,
            QuantityName qtyName, double qtyMin, double qtyMax, 
            double deltaQty)
        {
            this.adjustableDevice = adjustableDevice;
            this.qtyName = qtyName;
            this.qtyMin = qtyMin;
            this.qtyMax = qtyMax;
            this.deltaQty = deltaQty;
        }
        public void Press(int nTimes)
        {
            currentLevel += nTimes*deltaQty;
            if (currentLevel > qtyMax) currentLevel -= qtyMax;
            if (currentLevel < qtyMin) currentLevel += qtyMin;
            adjustableDevice.SetLevel(currentLevel, qtyName);
        }
    }
    class AnalogFMTuner
    {
        public Button powerButton;
        public AdjustingWheel freqWheel;
        public AdjustingWheel volWheel;
        private TunerUnit tunerUnit;
        public AnalogFMTuner()
        {
            tunerUnit = new TunerUnit();
            powerButton = new Button(tunerUnit);
            freqWheel = new AdjustingWheel(tunerUnit, 
                QuantityName.FREQUENCY, 80.0, 120.0);
            volWheel = new AdjustingWheel(tunerUnit, 
                QuantityName.VOLUME, 0.0, 12.0);
        }
        public bool PowerOn()
        {
            return tunerUnit.PowerOn();
        }
        public double Frequency
        {
            get { return tunerUnit.Frequecy; }
        }
        public double Volume
        {
            get { return tunerUnit.Volume; }
        }
    }
    class DigitalFMTuner
    {
        public Button powerButton;
        public AdjustingButton freqButton;
        public AdjustingButton volButton;
        private TunerUnit tunerUnit;
        public DigitalFMTuner()
        {
            tunerUnit = new TunerUnit();
            powerButton = new Button(tunerUnit);
            freqButton = new AdjustingButton(tunerUnit, 
                QuantityName.FREQUENCY, 80.0, 120.0, 0.01);
            volButton = new AdjustingButton(tunerUnit, 
                QuantityName.VOLUME, 0.0, 12.0, 0.5);
        }
        public bool PowerOn()
        {
            return tunerUnit.PowerOn();
        }
        public double Frequency
        {
            get { return tunerUnit.Frequecy; }
        }
        public double Volume
        {
            get { return tunerUnit.Volume; }
        }

    }       
    class Program
    {
        static void Main(string[] args)
        {
            const double TWO_PI = 2.0*3.141592653;
            AnalogFMTuner tunerA = new AnalogFMTuner();
            tunerA.powerButton.Press();
            Debug.Assert(tunerA.PowerOn()); 
            double angle = 0.1 * TWO_PI;
            tunerA.freqWheel.Turn(angle);
            Debug.Assert(tunerA.Frequency - ((120-80)*0.1 + 80) < 1.0e-7);
            angle = 0.2 * TWO_PI;
            tunerA.volWheel.Turn(angle);
            Debug.Assert(tunerA.Volume - 12*0.2 < 1.0e-7);
            tunerA.powerButton.Release();
            Debug.Assert(!tunerA.PowerOn());

            DigitalFMTuner tunerD = new DigitalFMTuner();
            tunerD.powerButton.Press();
            Debug.Assert(tunerD.PowerOn());
            tunerD.freqButton.Press(5);
            Debug.Assert(tunerD.Frequency - (80 + 0.02*5) < 1.0e-7);
            tunerD.volButton.Press(2);
            Debug.Assert(tunerD.Volume - 0.5*2 < 1.0e-7);
            tunerD.powerButton.Release();
            Debug.Assert(!tunerD.PowerOn());
        }
    }
}
