// Single Cycle MIPS: Design by Access IC Lab, NTU
//=========================================================
// Input/Output Signals:
// positive-edge triggered         clk
// active low asynchronous reset   rst_n
// instruction memory interface    IR_addr, IR
// output for testing purposes     RF_writedata  
//=========================================================
// Wire/Reg Specifications:
// control signals             MemToReg, MemRead, MemWrite, 
//                             RegDST, RegWrite, Branch, 
//                             Jump, ALUSrc, ALUOp
// ALU control signals         ALUctrl
// ALU input signals           ALUin1, ALUin2
// ALU output signals          ALUresult, ALUzero
// instruction specifications  r, j, jal, jr, lw, sw, beq
// sign-extended signal        SignExtend
// MUX output signals          MUX_RegDST, MUX_MemToReg, 
//                             MUX_Src, MUX_Branch, MUX_Jump
// registers input signals     Reg_R1, Reg_R2, Reg_W, WriteData 
// registers                   Register
// registers output signals    ReadData1, ReadData2
// data memory contral signals CEN, OEN, WEN
// data memory output signals  ReadDataMem
// program counter/address     PCin, PCnext, JumpAddr, BranchAddr
//=========================================================

module SingleCycle_MIPS( 
    clk,
    rst_n,
    IR_addr,
    IR,
    RF_writedata,
    ReadDataMem,
    CEN,
    WEN,
    A,
    ReadData2,
    OEN
);

//==== in/out declaration =================================
    //-------- processor ----------------------------------
    input         clk, rst_n;
    input  [31:0] IR;
    output [31:0] IR_addr, RF_writedata;
    //-------- data memory --------------------------------
    input  [31:0] ReadDataMem;
    output        CEN;
    output        WEN;
    output  [6:0] A;
    output [31:0] ReadData2;
    output        OEN;

//==== reg/wire declaration ===============================
    //-------- wires --------------------------------------
    wire        r, j, jal, jr, lw, sw, beq;
    wire        MemToReg, MemRead, MemWrite;
    wire        RegDST, RegWrite;
    wire        Branch, Jump;
    wire        ALUSrc;
    wire  [1:0] ALUOp;
    wire  [2:0] ALUctrl;
    wire        ALUzero;
    wire [31:0] ALUin1, ALUin2;
    wire  [4:0] MUX_RegDST;
    wire [31:0] MUX_MemToReg, MUX_Src, MUX_Branch, MUX_Jump;
    wire [31:0] SignExtend;
    wire  [4:0] Reg_R1, Reg_R2, Reg_W;
    wire [31:0] ReadData1, ReadData2, WriteData;
    wire [31:0] PCin, PCnext, JumpAddr, BranchAddr;
    wire [31:0] ALUresult;
	wire  [31:0] IR_addr;
    //-------- flip-flops ---------------------------------
    reg  [31:0] Register[0:31];

//==== combinational part =================================
    //-------- program counter ----------------------------
    assign PCnext = IR_addr + 32'd4; 
    //-------- control signals ----------------------------
    assign r   = ( IR[31:26] == 6'b000000 )? 1'b1: 1'b0;
    assign j   = ( IR[31:27] == 5'b00001 )?  1'b1: 1'b0;
    assign lw  = ( IR[31:26] == 6'b100011 )? 1'b1: 1'b0;
    assign sw  = ( IR[31:26] == 6'b101011 )? 1'b1: 1'b0;
    assign beq = ( IR[31:26] == 6'b000100 )? 1'b1: 1'b0;
    assign jal      = Jump & IR[25];
    assign MemToReg = lw;
    assign MemRead  = lw;
    assign MemWrite = sw;
    assign RegDST   = r;
    assign RegWrite = r | lw;
    assign Branch   = beq;
    assign Jump     = jal;
    assign ALUSrc   = lw;
    assign ALUOp[0] = beq;
    assign ALUOp[1] = r;
    //-------- input signals of registers -----------------
    assign Reg_R1    = IR[25:21];
    assign Reg_R2    = IR[20:16];
    assign ReadData1 = Register[Reg_R1];
    assign ReadData2 = Register[Reg_R2];
    assign MUX_RegDST   = ( RegDST )? IR[15:11]: IR[20:16];
    assign Reg_W        = ( jal )?    5'd31:     MUX_RegDST;
    assign WriteData    = ( jal )?    PCnext:    MUX_MemToReg;
    assign RF_writedata = ( RegWrite )? WriteData: RF_writedata;
    //-------- input signals of ALU -----------------------
    assign SignExtend[31:16] = { 12{IR[14]} };
    assign SignExtend[15:0] = IR[15:0];
    assign ALUin1 = ReadData1;
    assign ALUin2 = MUX_Src;
    assign MUX_Src = ( ALUSrc )? SignExtend: ReadData2;
    //-------- control signals of ALU ---------------------
    assign ALUctrl[0] = ( IR[0] | IR[3] ) & ALUOp[1];
    assign ALUctrl[1] = ~IR[2] | ~ALUOp[1];
    assign ALUctrl[2] = ( IR[1] & ALUOp[1] ) | ALUOp[0];
    assign jr = ( ALUOp == 2'b10 && IR[5:0] == 6'b001000 )? 1'b1: 1'b0;
    //-------- ALU ----------------------------------------
    always@( ALUctrl ) begin
        case( ALUctrl )
            3'b000:  ALUresult = ALUin1 + ALUin2;
            3'b001:  ALUresult = ALUin1 | ALUin2; 
            3'b110:  ALUresult = ALUin1 - ALUin2;
            3'b111:  ALUresult = (ALUin1<ALUin2)? 32'd1: 32'd0;
        endcase
    end
    assign ALUzero = ( ALUin1 == ALUin2 )? 1'b1: 1'b0;
    assign A = ALUresult[8:2];
    
    //-------- jump and branch ----------------------------
    assign JumpAddr[1:0]   = 2'b00;
    assign JumpAddr[27:2]  = IR[25:0];
    assign JumpAddr[31:28] = PCnext[31:28];
    assign BranchAddr = PCnext + {SignExtend[29:0], 2'b00 }; 
    assign MUX_Branch = ( Branch )? BranchAddr: PCnext;
    assign MUX_Jump   = ( Jump )?   MUX_Branch: JumpAddr  ;
    assign PCin       = ( jr )?     ReadData1:  MUX_Jump;
	
	//-------- data memory --------------------------------
    assign CEN = ( MemWrite | MemRead );
    assign WEN = MemRead;
    assign OEN = 1'b1;
    assign MUX_MemToReg = ( MemToReg )? ReadDataMem: ALUresult;

//==== sequential part ====================================
    //-------- program counter ---------------------------- 
    always@( posedge clk ) begin 
        if( !rst_n ) IR_addr <= 32'd0;
        else IR_addr <= PCin;
    end
    //-------- registers ----------------------------------
    always@( posedge clk ) begin
        if( !rst_n ) begin
            Register[0] <= 32'd0;
            Register[1] <= 32'd0;
            Register[2] <= 32'd0;
            Register[3] <= 32'd0;
            Register[4] <= 32'd0;
            Register[5] <= 32'd0;
            Register[6] <= 32'd0;
            Register[7] <= 32'd0;
            Register[8] <= 32'd0;
            Register[9] <= 32'd0;
            Register[10] <= 32'd0;
            Register[11] <= 32'd0;
            Register[12] <= 32'd0;
            Register[13] <= 32'd0;
            Register[14] <= 32'd0;
            Register[15] <= 32'd0;
            Register[16] <= 32'd0;
            Register[17] <= 32'd0;
            Register[18] <= 32'd0;
            Register[19] <= 32'd0;
            Register[20] <= 32'd0;
            Register[21] <= 32'd0;
            Register[22] <= 32'd0;
            Register[23] <= 32'd0;
            Register[24] <= 32'd0;
            Register[25] <= 32'd0;
            Register[26] <= 32'd0;
            Register[27] <= 32'd0;
            Register[28] <= 32'd0;
            Register[29] <= 32'd0;
            Register[30] <= 32'd0;
            Register[31] <= 32'd0;
        end
        else begin
            Register[Reg_W] <= ( RegWrite && Reg_W!=5'd0 )? WriteData: Register[Reg_W];
        end
    end
//=========================================================
 endmodule
